C      *****************************************************************
C      * Subroutine FSolub                                             *
C      * Calculates the amount of each component that is soluble and   *
C      * determines the fraction of that soluble total that will move  *
C      * with the flux.                                                *
C      *                                                               *
C      * Variables:                                                    *
C      * C         -Local - Component index.  Unitless.                *
C      * CtoM      -Fcn   - Multiplication factor for conversion from  *
C      *                    concentration to mass.  Units are L.       *
C      * DS        -Local - Dependent species index in arrays that     *
C      *                    contain components and dependent species.  *
C      *                    Unitless.                                  *
C      * Flux      -Input - The amount of water moved in the current   *
C      *                    time step.  A positive number means the    *
C      *                    flux is moving down.  A negative number    *
C      *                    means the flux is moving up.  Units are L. *
C      * LNum      -Input - The layer number from which the components *
C      *                    are moving.  Unitless.                     *
C      * MovComp   -Output- A matrix, of size NComp by NLayers+1,      *
C      *                    that contains the amount of each component *
C      *                    moved in each flux.  If the amount is      *
C      *                    positive the component moves from the layer*
C      *                    with the same number as the flux to layer  *
C      *                    flux+1.  If the amount is negative the     *
C      *                    component moves from layer flux to flux-1. *
C      *                    Units are mol.                             *
C      *                    (Common block VModel, file VModel.f)       *
C      * NComp     -Input - The number of components in the system.    *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NSpecies  -Input - The number of species in the system (i.e.  *
C      *                    the number of components plus the number   *
C      *                    of dependent species).  Unitless.          *
C      *                    (Common block VModel, file VModel.f)       *
C      * SC        -Input - A matrix, of size NDepSp by NComp, of the  *
C      *                    stoichiometric coefficient of each         *
C      *                    component for each species.  Unitless.     *
C      *                    (Common block VModel, file VModel.f)       *
C      * SoilWat   -Input - The soil water content.  Units are L.      *
C      * SpConc    -Input - A matrix, of size NSpecies by NLayers, of  *
C      *                    species concentrations, one set for each   *
C      *                    layer. Units are mol/L.                    *
C      *                    (Common block VModel, file VModel.f)       *
C      * ST        -Local - The soluble total for the current          *
C      *                    component.  Units are mol.                 *
C      * SType     -Input - A vector, of size NSpecies, of each        *
C      *                    species's type.  Unitless.                 *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE FSOLUB(LNUM, FLUX, SOILWAT)
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'
							INTEGER C, DS, LNUM
							REAL*8 CTOM, FLUX, SOILWAT, ST
							
C      *--------------------------*
C      * Loop over all components *
C      *--------------------------*
       DO 100 C=1, NCOMP
	      
C         *----------------------------------------------------------*
C         * If the component is aqueous then initialize the soluble  *
C         * total to the amount of the component as a free component,*
C         * otherwise initialize the soluble total to zero.  Then    *
C         * add in the contributions from the aqueous dependent      *
C         * species.                                                 *
C         *----------------------------------------------------------*
          IF (STYPE(C).LT.SPMASBAL) THEN
	      	     ST = SPCONC(C,LNUM) * CTOM(C,LNUM)
      		  ELSE
      		     ST = 0
      		  ENDIF
		  
          DO 200 DS=NCOMP+1, NSPECIES
         			 IF (STYPE(DS).LT.SPMASBAL) THEN
	      	        ST = ST + SPCONC(DS,LNUM) * CTOM(DS,LNUM) * 
     >               SC(DS,C)
		           ENDIF
  200     CONTINUE

C         *------------------------------------------------------*
C         * Calculate the fraction of this soluble total that is *
C         * moved with the flux as the fraction of the flux to   *
C         * the soil water.                                      *
C         *------------------------------------------------------*
          IF (SOILWAT.EQ.0.0) THEN
             MOVCOMP(C,LNUM) = 0.0
          ELSE
             MOVCOMP(C,LNUM) = ST * FLUX / SOILWAT
          ENDIF
  100  CONTINUE

       RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
